/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.boot.model.source.spi;

import org.hibernate.tuple.GenerationTiming;

/**
 * Source-agnostic description of information needed to bind a singular attribute.
 *
 * @author Steve Ebersole
 */
public interface SingularAttributeSource extends AttributeSource {
	/**
	 * Determine whether this is a virtual attribute or whether it physically exists on the users domain model.
	 *
	 * @return {@code true} indicates the attribute is virtual, meaning it does NOT exist on the domain model;
	 *         {@code false} indicates the attribute physically exists.
	 */
	public boolean isVirtualAttribute();

	/**
	 * Obtain the nature of this attribute type.
	 *
	 * @return The attribute type nature
	 */
	public SingularAttributeNature getSingularAttributeNature();

	/**
	 * Obtain a description of if/when the attribute value is generated by the database.
	 *
	 * @return The attribute value generation information
	 */
	public GenerationTiming getGenerationTiming();

	/**
	 * Did the mapping specify that the given attribute value(s) should be inserted into the database?
	 *
	 * @return {@code true} indicates value(s) should be inserted; {@code false} indicates not.
	 */
	Boolean isInsertable();

	/**
	 * Did the mapping specify that the given attribute value(s) should be updated in the database?
	 *
	 * @return {@code true} indicates value(s) should be updated; {@code false} indicates not.
	 */
	Boolean isUpdatable();

	/**
	 * Should the attribute be  lazily loaded by bytecode enhancement?
	 *
	 * @return {@code true} to indicate the attribute should be lazily loaded by bytecode enhancement?
	 */
	public boolean isBytecodeLazy();

	/**
	 * Retrieve the natural id mutability
	 *
	 * @return The mutability, see enum for meanings
	 */
	public NaturalIdMutability getNaturalIdMutability();

}
